/**
 * Extracts a Google Drive file ID from all known public share URL formats:
 *
 * - https://drive.google.com/file/d/{ID}/view
 * - https://drive.google.com/file/d/{ID}/view?usp=sharing
 * - https://drive.google.com/open?id={ID}
 * - https://drive.google.com/uc?id={ID}&export=download
 * - https://drive.google.com/uc?export=download&id={ID}
 * - https://drive.google.com/drive/folders/{ID}  (folder — rejected)
 * - https://docs.google.com/document/d/{ID}/edit
 * - https://docs.google.com/spreadsheets/d/{ID}/edit
 * - https://docs.google.com/presentation/d/{ID}/edit
 * - Bare file ID (32+ alphanumeric characters)
 */
export function extractDriveFileId(url: string): string | null {
  if (!url || typeof url !== "string") return null;

  const trimmed = url.trim();

  // 1. Handle bare file IDs (no URL structure) — Drive IDs are ~33 chars, alphanumeric + dash/underscore
  if (/^[a-zA-Z0-9_-]{25,}$/.test(trimmed)) {
    return trimmed;
  }

  try {
    const parsedUrl = new URL(trimmed);
    const hostname = parsedUrl.hostname;

    // Must be a Google domain
    if (!hostname.endsWith("google.com")) {
      return null;
    }

    // 2. /file/d/{ID}/... pattern (drive.google.com and docs.google.com variants)
    const filePathMatch = parsedUrl.pathname.match(
      /\/(?:file|document|spreadsheets|presentation|forms)\/d\/([a-zA-Z0-9_-]{25,})(?:\/|$)/,
    );
    if (filePathMatch?.[1]) {
      return filePathMatch[1];
    }

    // 3. ?id={ID} query param pattern (open, uc, export links)
    const idParam = parsedUrl.searchParams.get("id");
    if (idParam && /^[a-zA-Z0-9_-]{25,}$/.test(idParam)) {
      return idParam;
    }

    return null;
  } catch {
    return null;
  }
}
