import { StorageProvider } from "./storage-provider.interface";
import { Readable } from "stream";
import axios from "axios";

export class PixeldrainProvider implements StorageProvider {
  private readonly apiUrl = "https://pixeldrain.com/api";
  private readonly apiKey: string;

  constructor() {
    this.apiKey = process.env.PIXELDRAIN_API_KEY || "";
  }

  async uploadStream(
    stream: Readable,
    meta: { name: string; mimeType: string; size?: number },
  ): Promise<{ identifier: string; url: string } | null> {
    try {
      const url = `${this.apiUrl}/file/${encodeURIComponent(meta.name)}`;

      const config: any = {
        headers: {
          "Content-Type": meta.mimeType || "application/octet-stream",
        },
      };

      if (this.apiKey) {
        config.auth = {
          username: "",
          password: this.apiKey,
        };
      }

      const response = await axios.put(url, stream, config);

      if (response.data && response.data.id) {
        const publicUrl = `https://pixeldrain.com/api/file/${response.data.id}`;
        return { identifier: response.data.id, url: publicUrl };
      }
      return null;
    } catch (error: any) {
      console.error("[Pixeldrain Upload] HTTP Error:", error?.message);
      if (error?.response) {
        console.error(
          "[Pixeldrain Upload] Error Status:",
          error.response?.status,
        );
        console.error("[Pixeldrain Upload] Error Data:", error.response?.data);
      }
      throw new Error("Failed to upload to Pixeldrain");
    }
  }

  async deleteFile(identifier: string): Promise<boolean> {
    if (!this.apiKey) {
      console.warn(
        "Pixeldrain delete skipped: No API key provided (anonymous upload).",
      );
      return false;
    }

    try {
      const url = `${this.apiUrl}/file/${identifier}`;

      const response = await axios.delete(url, {
        auth: {
          username: "",
          password: this.apiKey,
        },
      });

      return response.data && response.data.success;
    } catch (error: any) {
      console.error(
        `[Pixeldrain Delete] error for ${identifier}:`,
        error?.message,
      );
      return false;
    }
  }
}
