import { Request, Response } from "express";
import { FileService } from "../services/file.service";

export const uploadFile = async (
  req: Request,
  res: Response,
): Promise<void> => {
  try {
    const { url } = req.body;
    if (!url) {
      res.status(400).json({ error: 'Missing "url" parameter in body' });
      return;
    }

    const fileRecord = await FileService.uploadFromDrive(url);
    res.status(202).json({
      message: "Upload started asynchronously",
      file: formatFileRecord(fileRecord),
    });
  } catch (error: any) {
    res.status(400).json({ error: error.message });
  }
};

export const listFiles = async (req: Request, res: Response): Promise<void> => {
  try {
    const files = await FileService.getAllFiles();
    res.status(200).json(files.map(formatFileRecord));
  } catch (error: any) {
    res.status(500).json({ error: "Internal server error" });
  }
};

export const getFile = async (req: Request, res: Response): Promise<void> => {
  try {
    const id = req.params.id as string;
    const file = await FileService.getFile(id);
    if (!file) {
      res.status(404).json({ error: "File not found" });
      return;
    }
    res.status(200).json(formatFileRecord(file));
  } catch (error: any) {
    res.status(500).json({ error: "Internal server error" });
  }
};

export const deleteFile = async (
  req: Request,
  res: Response,
): Promise<void> => {
  try {
    const id = req.params.id as string;
    await FileService.deleteFile(id);
    res.status(200).json({ message: "File deleted successfully" });
  } catch (error: any) {
    res.status(404).json({ error: error.message });
  }
};

// BigInt formatting helper for JSON responses
function formatFileRecord(record: any) {
  return {
    ...record,
    size: record.size ? record.size.toString() : null,
  };
}
