import { Request, Response } from "express";
import { prisma } from "../index";
import {
  DownloadService,
  DownloadProvider,
} from "../services/download.service";

export class DownloadController {
  /**
   * Returns file metadata along with all available provider download sources.
   * GET /api/download/:id
   */
  static async getDownloadInfo(req: Request, res: Response): Promise<void> {
    try {
      const id = String(req.params.id);

      const file = await prisma.file.findUnique({ where: { id } });

      if (!file) {
        res.status(404).json({ error: "File not found" });
        return;
      }

      if (file.status !== "COMPLETED") {
        res.status(409).json({
          error: "File is not ready for download",
          status: file.status,
        });
        return;
      }

      const providers: Record<string, boolean> = {
        r2: !!file.r2Key,
        pixeldrain: !!file.pixeldrainId,
        idrive: !!file.idriveKey,
        vikingfile: !!file.vikingfileId,
      };

      res.status(200).json({
        id: file.id,
        originalName: file.originalName,
        mimeType: file.mimeType,
        size: file.size ? file.size.toString() : null,
        providers,
        // VikingFile browse page URL — opened directly in a new tab on the frontend
        vikingfileUrl: file.vikingfileUrl ?? null,
      });
    } catch (error) {
      console.error("[DownloadController] getDownloadInfo error:", error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  }

  /**
   * Proxy-streams a file from the requested provider through our backend.
   * GET /api/download/:id/proxy?provider=r2|pixeldrain|idrive|vikingfile
   */
  static async proxyDownload(req: Request, res: Response): Promise<void> {
    try {
      const id = String(req.params.id);
      const provider = String(
        Array.isArray(req.query.provider)
          ? req.query.provider[0]
          : req.query.provider || "",
      ).toLowerCase() as DownloadProvider;

      const validProviders: DownloadProvider[] = [
        "r2",
        "pixeldrain",
        "idrive",
        "vikingfile",
      ];
      if (!provider || !validProviders.includes(provider)) {
        res.status(400).json({
          error: `Invalid provider. Must be one of: ${validProviders.join(", ")}`,
        });
        return;
      }

      const file = await prisma.file.findUnique({ where: { id } });

      if (!file) {
        res.status(404).json({ error: "File not found" });
        return;
      }

      if (file.status !== "COMPLETED") {
        res.status(409).json({ error: "File is not ready for download" });
        return;
      }

      // Validate the requested provider actually has data
      const identifierMap: Record<DownloadProvider, string | null> = {
        r2: file.r2Key,
        pixeldrain: file.pixeldrainId,
        idrive: file.idriveKey,
        vikingfile: file.vikingfileId,
      };
      const storedUrlMap: Record<DownloadProvider, string | null> = {
        r2: file.r2Url,
        pixeldrain: file.pixeldrainUrl,
        idrive: file.idriveUrl,
        vikingfile: file.vikingfileUrl,
      };

      const identifier = identifierMap[provider];
      if (!identifier) {
        res
          .status(404)
          .json({ error: `File is not hosted on provider: ${provider}` });
        return;
      }

      const filename = file.originalName || "download";
      const mimeType = file.mimeType || "application/octet-stream";

      const result = await DownloadService.resolveStream(
        {
          provider,
          identifier,
          storedUrl: storedUrlMap[provider] ?? undefined,
        },
        filename,
        mimeType,
      );

      // Set download headers
      const safeFilename = encodeURIComponent(filename);
      res.setHeader("Content-Type", result.contentType);
      res.setHeader(
        "Content-Disposition",
        `attachment; filename="${filename}"; filename*=UTF-8''${safeFilename}`,
      );
      res.setHeader("Cache-Control", "no-store");
      if (result.contentLength) {
        res.setHeader("Content-Length", result.contentLength);
      }

      result.stream.pipe(res);

      result.stream.on("error", (err) => {
        console.error(
          `[DownloadController] Stream error for file ${id} via ${provider}:`,
          err,
        );
        if (!res.headersSent) {
          res
            .status(502)
            .json({ error: "Failed to stream file from provider" });
        }
      });
    } catch (error) {
      console.error("[DownloadController] proxyDownload error:", error);
      if (!res.headersSent) {
        res.status(500).json({ error: "Internal Server Error" });
      }
    }
  }
}
