# 📦 File Upload Backend

> A production-ready Node.js + Express API that **streams files directly from Google Drive to Cloudflare R2** — no local disk storage required. Supports arbitrarily large files via multipart streaming.

---

## 🏗️ Architecture

```
Client → POST /files/upload (Google Drive URL)
           ↓ (immediate 202 response)
       Background Worker
           ↓ Fetch metadata (Google Drive API v3)
           ↓ Stream file bytes (Google Drive → Node.js → R2)
           ↓ Update DB status (PENDING → DOWNLOADING → COMPLETED)
Client ← Polls GET /files/:id for progress
```

- **No local storage** — bytes are piped directly from Google Drive to R2 using Node.js streams
- **Async processing** — API responds immediately, upload continues in the background
- **Progress tracking** — DB is updated as multipart chunks are uploaded to R2

---

## 🧰 Tech Stack

| Layer            | Technology                    |
| ---------------- | ----------------------------- |
| Runtime          | Node.js 22                    |
| Framework        | Express 5                     |
| Language         | TypeScript 5                  |
| ORM              | Prisma 7 + PostgreSQL adapter |
| Database         | PostgreSQL (Neon recommended) |
| Storage          | Cloudflare R2 (S3-compatible) |
| File source      | Google Drive API v3           |
| Package manager  | pnpm                          |
| Containerisation | Docker (multi-stage)          |

---

## 📁 Project Structure

```
backend/
├── src/
│   ├── config/
│   │   └── r2.config.ts          # S3 client for Cloudflare R2
│   ├── controllers/
│   │   └── file.controller.ts    # HTTP request handlers
│   ├── routes/
│   │   └── file.routes.ts        # Express router
│   ├── services/
│   │   └── file.service.ts       # Business logic + streaming
│   ├── utils/
│   │   └── google-drive.utils.ts # Drive URL → file ID parser
│   └── index.ts                  # App entry point
├── prisma/
│   ├── schema.prisma             # Database schema
│   └── migrations/               # Migration history
├── prisma.config.ts              # Prisma v7 configuration
├── Dockerfile                    # Multi-stage production build
├── .dockerignore
└── .env                          # Environment variables (not committed)
```

---

## 🔌 API Endpoints

| Method   | Path            | Description                                     |
| -------- | --------------- | ----------------------------------------------- |
| `POST`   | `/files/upload` | Start background upload from a Google Drive URL |
| `GET`    | `/files`        | List all files with status and progress         |
| `GET`    | `/files/:id`    | Get a specific file record                      |
| `DELETE` | `/files/:id`    | Delete file from R2 and the database            |
| `GET`    | `/health`       | Health check                                    |

### `POST /files/upload`

**Request body:**

```json
{ "url": "https://drive.google.com/file/d/FILE_ID/view" }
```

**Response `202`:**

```json
{
  "id": "uuid",
  "status": "PENDING",
  "createdAt": "2026-02-26T..."
}
```

### File Status Lifecycle

```
PENDING → DOWNLOADING → COMPLETED
                    ↘ FAILED
```

---

## 🗄️ Database Schema

```prisma
model File {
  id           String    @id @default(uuid())
  originalName String?
  mimeType     String?
  size         BigInt?           // supports files > 2GB
  status       String    @default("PENDING")
  progress     Int       @default(0)
  r2Key        String?           // key inside R2 bucket
  createdAt    DateTime  @default(now())
  updatedAt    DateTime  @updatedAt
}
```

---

## ⚙️ Environment Variables

Create a `.env` file in the `backend/` directory:

```env
# Server
PORT=3030

# Database (PostgreSQL / Neon)
DATABASE_URL="postgresql://user:password@host:5432/dbname"

# Cloudflare R2
R2_ACCOUNT_ID="your_cloudflare_account_id"
R2_ACCESS_KEY_ID="your_r2_access_key_id"
R2_SECRET_ACCESS_KEY="your_r2_secret_access_key"
R2_BUCKET_NAME="your_bucket_name"

# Google Drive API
GOOGLE_DRIVE_API_KEY="your_google_api_key"
```

### How to get credentials

| Variable                    | Where to find it                                     |
| --------------------------- | ---------------------------------------------------- |
| `DATABASE_URL`              | Neon dashboard → Connection string                   |
| `R2_ACCOUNT_ID`             | Cloudflare dashboard → right sidebar                 |
| `R2_ACCESS_KEY_ID / SECRET` | Cloudflare R2 → Manage API Tokens → Create Token     |
| `GOOGLE_DRIVE_API_KEY`      | Google Cloud Console → APIs & Services → Credentials |

> **Note:** Google Drive API must be enabled in your Google Cloud project. Only **publicly shared** Drive files are supported.

---

## 🚀 Getting Started

### Prerequisites

- Node.js 22+
- pnpm
- PostgreSQL database
- Cloudflare R2 bucket
- Google Drive API key

### Local Development

```bash
# Install dependencies
pnpm install

# Push schema to database
pnpm exec prisma db push

# Start development server (with hot reload)
pnpm run dev
```

Server starts at `http://localhost:3030`

### Production Build

```bash
pnpm run build           # Compile TypeScript → dist/
pnpm run migrate:prod    # Apply pending migrations
pnpm run start           # Start compiled server
# or combined:
pnpm run start:prod      # migrate:prod + start
```

---

## 🐳 Docker Deployment

### Build image

```bash
docker build -t file-upload-backend .
```

### Run container

```bash
docker run -p 3030:3030 \
  -e DATABASE_URL="..." \
  -e R2_ACCOUNT_ID="..." \
  -e R2_ACCESS_KEY_ID="..." \
  -e R2_SECRET_ACCESS_KEY="..." \
  -e R2_BUCKET_NAME="..." \
  -e GOOGLE_DRIVE_API_KEY="..." \
  file-upload-backend
```

The container automatically runs `prisma migrate deploy` before starting the server.

### Coolify / Self-hosted

1. Point your service at this repository
2. Set **Dockerfile path** to `Dockerfile`
3. Add all environment variables in the Coolify dashboard
4. Deploy — migrations run automatically on startup

---

## 📜 Scripts

| Script                  | Description                                          |
| ----------------------- | ---------------------------------------------------- |
| `pnpm run dev`          | Start dev server with hot reload (nodemon + ts-node) |
| `pnpm run build`        | Compile TypeScript to `dist/`                        |
| `pnpm run start`        | Run compiled production server                       |
| `pnpm run start:prod`   | Run migrations then start server                     |
| `pnpm run migrate:prod` | Apply pending DB migrations (production-safe)        |
